<?php
declare(strict_types=1);

namespace Lookways\SDK\Api;

use Lookways\SDK\Client;
use Lookways\SDK\Exception\BadRequestException;
use Lookways\SDK\Exception\ForbidenException;
use Lookways\SDK\Exception\LookWaysBackendException;
use Lookways\SDK\Exception\WrongCodeException;
use Lookways\SDK\Exception\WrongCredentialTypeException;

abstract class Api
{





    public function __construct(protected Client $sdkClient) {

    }



    public function getSDKClient() : Client
    {
        return $this->sdkClient;
    }



    public function request(string $endpoint, string $method, array $data, array $files = [], bool $allowJson = true, bool $credentialsRequired=true): array|string
    {

        $curl = curl_init();

        curl_reset($curl);

        $method = strtoupper($method);

        curl_setopt_array($curl, [

            CURLOPT_URL =>  $this->sdkClient->getBackendUrl() . '/' . $endpoint,
            CURLOPT_RETURNTRANSFER => true,


        ]);


        if($credentialsRequired)
        {
                $bearer = $this->sdkClient->getBearer();
        } else
        {
            $bearer = null;
        }
        $headers = [];

        if(!empty($bearer))
        {
            $headers[] = 'Authorization: Bearer '.$bearer;
        }
        if(!empty($data))
        {
            if ($method == "POST" || $method == 'PATCH') {
                curl_setopt($curl, CURLOPT_POST, 1);
                if (empty($files)) {

                    if ($allowJson) {
                        $data_string = json_encode($data);
                        curl_setopt($curl, CURLOPT_POSTFIELDS, $data_string);
                        $headers[] = 'Content-Type: application/json';
                        $headers[] = 'Content-Length: ' . strlen($data_string);
                    } else
                        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);

                } else {

                    foreach($files as $key => $value)
                    {

                            $data[$key] = new \CURLFile($value);
                    }
                    curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
                }
            } else
                if($method == "GET")
                {

                    curl_setopt($curl, CURLOPT_URL, $this->sdkClient->getBackendUrl() . '/' . $endpoint.'?'.http_build_query(
                            $data
                        ));

                } else {
                    curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
                    curl_setopt($curl, CURLOPT_POSTFIELDS, $data);



                }
        }

        if ($method == 'DELETE')
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "DELETE");
        if ($method == 'PATCH')
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'PATCH');

        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($curl, CURLOPT_TIMEOUT, 10);


        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);

        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);

        $response = curl_exec($curl);

        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        $contentType = curl_getinfo($curl, CURLINFO_CONTENT_TYPE);

        $error = curl_error($curl);



        curl_close($curl);




        if($httpCode >= 300)
        {
            if($httpCode == 401)
            {
                throw new ForbidenException($response);
            }
            if($httpCode == 400)
            {
                throw new BadRequestException($response);
            }
            throw new WrongCodeException(json_encode($data) . ' ' . $response . ' ' . $httpCode);

        }



        if(!empty($error))
        {
            throw new LookWaysBackendException($error);
        }




        $data = $response;
        if(is_string($contentType)) {
            if (str_contains($contentType, "application/json"))
                $data = json_decode($response, true);
        }


        return $data;
    }



    public function get(string $endpoint, array $data, bool $authenticated=false) : array|string
    {

        return $this->request($endpoint, 'GET', $data, credentialsRequired: $authenticated);

    }




    public function post(string $endpoint, array $data, array $files = [], bool $authenticated=true): array|string
    {
        return $this->request($endpoint, 'POST', $data, $files, true, credentialsRequired: $authenticated);
    }


    public function patch(string $endpoint, array $data, array $files = [], bool $authenticated=true): array|string
    {
        return $this->request($endpoint, 'PATCH', $data, $files, credentialsRequired: $authenticated);
    }


    public function delete(string $endpoint, bool $authenticated=true): void
    {
        $this->request($endpoint, 'DELETE', [], credentialsRequired: $authenticated);
    }
}