<?php
declare(strict_types=1);


namespace PrestaShop\Module\Lookwaysplugin\Service;

use Doctrine\ORM\EntityManagerInterface;
use Lookways\SDK\Api\Variation;
use PrestaShop\Module\Lookwaysplugin\Entity\LookwaysProduct;

class LookwaysProductService
{


    protected Variation $variationApi;


    protected $lookwaysProductRepo;


    public function __construct(
        protected EntityManagerInterface $entityManager,
        protected \Context $context,
        protected ?LookwaysApiService $apiService = null

    )
    {

        if ($this->apiService === null)
            $this->apiService = new LookwaysApiService();

        $this->variationApi = $this->apiService->getSDKClient()->getVariationAPI();

        $this->lookwaysProductRepo = $this->entityManager->getRepository(LookwaysProduct::class);

    }



    public function updateProduct(\Product $product)
    {

        $mainEntity = $this->updateEntity($product);
        $productId = null;
        if($product->hasCombinations())
        {
            foreach($product->getAttributeCombinations($this->context->language->id) as $combinationData)
            {


                $combination = new \Combination($combinationData['id_product_attribute']);



                $combinationEntity = $this->updateEntity($combination);

                if(!empty($combinationEntity))
                {
                    $this->entityManager->detach($combinationEntity);
                    $productId = $combinationEntity->getLookwaysProductId();
                }







            }
        }


        if(empty($mainEntity))
        {

            $mainEntity = new LookwaysProduct();
            $mainEntity->setId($this->getLookwaysId($product));



            $mainEntity->setLookwaysProductId($productId);
            $this->entityManager->persist($mainEntity);
            $this->entityManager->flush();
        }


    }





    public function getProductCombinationsAttributes(\Product $product)
    {

        $combinations = [];

        foreach($product->getAttributeCombinations($this->context->language->id) as $combinationData)
        {
            $combinations[$combinationData['id_product_attribute']][] = $combinationData;
        }

        return $combinations;

    }


    public function getProductVariationsData(\Product $product) : ?array
    {

        if(!$product->hasCombinations())
            return null;

        $data = [];



        foreach($this->getProductCombinationsAttributes($product) as $combinationId => $combinationData) {


            $id = $this->prepareLookwaysId((string) $product->id, (string) $combinationId);

            $entity = $this->getLookwaysProductEntity($id);

            if(empty($entity))
                continue;





            $data[] = [
                'features' => join("|", array_map(function($item)  {
                    return $item['id_attribute_group'] . ':' . $item['id_attribute'];
                }, $combinationData)),
                'variationId' => $entity->getLookwaysVariationId(),
                'productId' => $entity->getLookwaysProductId(),
            ];

            $this->entityManager->detach($entity);


        }

        return $data;
    }


    protected function updateEntity(\Product | \Combination $item) : ?LookwaysProduct
    {

        $lookwaysData = $this->getLookwaysData($item);



        if($lookwaysData === null)
            return null;

        $id = $this->getLookwaysId($item);


        $entity = $this->getLookwaysProductEntity($id);

        if(empty($entity)) {
            $entity = new LookwaysProduct();
            $entity->setId($id);

        }

        $entity->setLookwaysProductId($lookwaysData['digiProduct']);
        $entity->setLookwaysVariationId($lookwaysData['id']);

        $this->entityManager->persist($entity);
        $this->entityManager->flush();

        //$this->updatePrice($lookwaysData['storeFrontCode'], intval(\Product::getPriceStatic($item->id_product, true, $item->id) * 100));

        $this->updateMeta($lookwaysData['storeFrontCode'], $item);

        return $entity;

    }


    protected function updatePrice(string $code, int $price)
    {

        $this->variationApi->updatePrice($code, $price);

    }


    protected function updateMeta(string $code, \Combination $item)
    {

        $link = new \Link();
        $prodUrl = $link->getProductLink($item->id_product);

        $this->variationApi->updateVariation($code, [
            'price' => intval(\Product::getPriceStatic($item->id_product, true, $item->id) * 100),
            'shopUrl' => $prodUrl . '?id_product_attribute=' . $item->id
        ]);
    }


    public function getLookwaysId(\Product | \Combination $item) : string
    {

        if($item instanceof \Product)
            return $this->prepareLookwaysId((string) $item->id, null);

        return $this->prepareLookwaysId((string) $item->id_product, (string) $item->id);

    }

    public function prepareLookwaysId(string $productId, ?string $combinationId)
    {
        $id = $productId;

        if(!empty($combinationId))
            $id .= '_' . $combinationId;

        return $id;
    }


    public function checkIfIntegrated(string $id)
    {
        $entity = $this->getLookwaysProductEntity($id);

        if(!empty($entity))
        {
            $this->entityManager->detach($entity);
            return true;
        }

        return false;

    }



    public function getLookwaysData(\Product | \Combination $item) : ?array
    {

        $codes = [
            $item->reference,
            $item->ean13,
            $item->isbn,
            $item->mpn,
            $item->upc
        ];



        $lookwaysEntityData = null;



        foreach($codes as $code) {

            if(empty($code))
                continue;

            try {
                $lookwaysEntityData = $this->variationApi->getVariationData($code);
            } catch(\Exception $exception)
            {

            }



            if(!empty($lookwaysEntityData))
                break;
        }

        if(!empty($lookwaysEntityData))
            return $lookwaysEntityData;

        return null;

    }

    public function getLookwaysProductEntity(string $id) : ?LookwaysProduct
    {
        if(empty($id))
            return null;
        return $this->lookwaysProductRepo->find($id);
    }






//    public function getLookwaysProductEntityByProductID(\Product $product, string $id)
//    {
//
//         $lookwaysLocalEntity = $this->entityManager->getRepository(LookwaysProduct::class)->find($id);
//
//         if(!empty($lookwaysLocalEntity))
//             return $lookwaysLocalEntity;
//
//
//         if($product->hasCombinations())
//         {
//
//             $combinations = $product->getAttributeCombinations($this->context->language->id);
//
//
//         }
//
//         $lookwaysLocalEntity = new LookwaysProduct();
//
//         $lookwaysLocalEntity->setId($id);
//
//         $codes = [
//             $product->ean13,
//             $product->isbn,
//             $product->mpn,
//             $product->upc
//         ];
//
//         $lookwaysEntityData = null;
//
//         $client = $this->apiService->getSDKClient()->getVariationAPI();
//
//         foreach($codes as $code) {
//
//             if(empty($code))
//                 continue;
//
//             $lookwaysEntityData = $client->getVariationData($code);
//
//             if(!empty($lookwaysEntityData))
//                 break;
//         }
//
//         $lookwaysLocalEntity->setLookwaysVariationId(intval($lookwaysEntityData['id']));
//         $lookwaysLocalEntity->setLookwaysProductId($lookwaysEntityData['digiProduct']);
//
//
//         $this->entityManager->persist($lookwaysLocalEntity);
//         $this->entityManager->flush();
//
//         return $lookwaysLocalEntity;
//
//    }


}

